
//****************************************************************************
//
// WIREWORKS PACKETS
// Packet.c
//
// By: The White Salmon Group, Inc.
// Copyright 2006
// All rights reserved
//
//
//****************************************************************************


#include <ansi_c.h>

#include <windows.h>
#include "FTD2XX.H"

#include "types.h"
#include "packet.h"

#define TIMS_ID			0xD0
#define HOST_ID			0x01

#define TIMEOUT_LOOPS	100000

#define DEBUGGING	0

#define USE_LRC		(version>=0xD0)

// sorry, only supports one device at a time
static FT_HANDLE handle=0;
static FT_STATUS status;

static int version=0;

// Forward references
static int SendBuffer(char* buffer, int nchars);
static int RecvBuffer(TIMS_packet* receive);
static void RecvFlush(void);
static void Show(char buffer[], int count);


// The TIMS device (PIC?) uses reverse endian
// swap bytes in place
U16 BYTE_SWAP(U16 a)
{
	union {
		U8 tmp[3];
		U16 var;
		} data;

	data.var = a;
	data.tmp[2] = data.tmp[0];
	data.tmp[0] = data.tmp[1];
	data.tmp[1] = data.tmp[2];
	return data.var;
}


int TIMS_Open(FT_HANDLE* pHandle, int ver/* version, temporary */)
{
	status = FT_Open(0, &handle);

	if (handle==NULL || status != FT_OK)
		return -1;
		
	version = ver;	// read this from device someday
	
	status = FT_ResetDevice(handle);
	status = FT_SetBaudRate(handle,FT_BAUD_230400);
	status = FT_SetDataCharacteristics(handle, FT_BITS_8,FT_STOP_BITS_1,FT_PARITY_NONE);
	status = FT_SetTimeouts(handle, 500, 100);
	*pHandle = handle;
	return 0;
}

int TIMS_Close(void)
{
	if (handle==0)
		return -1; 
	
	status = FT_Close(handle);
	handle = 0;
	return 0;
}


int TIMS_Ext5VOn(void)
{
	TIMS_packet transmit={0};
	TIMS_packet receive={0};
	int status;
	int count;
	
	transmit.to 	= TIMS_ID;
	transmit.from	= HOST_ID;
	transmit.ctrl	= BYTE_SWAP(EXT_5V_ON);
	transmit.len	= 0;
	RecvFlush();
	status = SendBuffer(transmit.buffer, 9+transmit.len);
	status = RecvBuffer(&receive);
	return status;
}	

int TIMS_Ext5VOff(void)
{
	TIMS_packet transmit={0};
	TIMS_packet receive={0};
	int status;
	int count;
	
	transmit.to 	= TIMS_ID;
	transmit.from	= HOST_ID;
	transmit.ctrl	= BYTE_SWAP(EXT_5V_OFF);
	transmit.len	= 0;
	RecvFlush();
	status = SendBuffer(transmit.buffer, 9+transmit.len);
	status = RecvBuffer(&receive);
	return (status);
}	

int TIMS_SPIConfigSet(U8 addr, U8 clock, U8 tvar)
{
	TIMS_packet transmit={0};
	TIMS_packet receive={0};
	int status;
	int count;

	transmit.to 	= TIMS_ID;
	transmit.from	= HOST_ID;
	transmit.ctrl	= BYTE_SWAP(SPI_DEVICE_CONFIG_WE);
	transmit.len	= 2;
	transmit.data[0] = addr;
	transmit.data[1] = clock;
	transmit.data[2] = tvar;
	RecvFlush();
	status = SendBuffer(transmit.buffer, 9+transmit.len);
	status = RecvBuffer(&receive);
	return (status);
}

int TIMS_SPIConfigGet(U8 addr, U8* clock, U8* tvar)
{
	TIMS_packet transmit={0};
	TIMS_packet receive={0};
	int status;
	int count;

	transmit.to 	= TIMS_ID;
	transmit.from	= HOST_ID;
	transmit.ctrl	= BYTE_SWAP(SPI_DEVICE_CONFIG_RE);
	transmit.len	= 2;
	transmit.data[0] = addr;
	RecvFlush();
	status = SendBuffer(transmit.buffer, 9+transmit.len);
	status = RecvBuffer(&receive);
	*clock = receive.data[0];
	*tvar  = receive.data[1];
	return (status);
}

int TIMS_SPIDeviceEnable(U8 addr)
{
	TIMS_packet transmit={0};
	TIMS_packet receive={0};
	int status;
	int count;

	transmit.to 	= TIMS_ID;
	transmit.from	= HOST_ID;
	transmit.ctrl	= BYTE_SWAP(SPI_DEVICE_ENABLE);
	transmit.len	= 0;
	transmit.data[0] = addr;
	RecvFlush();
	status = SendBuffer(transmit.buffer, 9+transmit.len);
	status = RecvBuffer(&receive);
	return (status);
}


int TIMS_SPIDataIO(U8 buffer[], int* len)
{
	TIMS_packet transmit={0};
	TIMS_packet receive={0};
	int status;
	int count;
	int length = *len;;

	if (length<1||length>256)
		return 5247;	// invalid length

	transmit.to 	= TIMS_ID;
	transmit.from	= HOST_ID;
	transmit.ctrl	= BYTE_SWAP(SPI_DATA_IO);
	transmit.len	= length-1;
	memcpy(transmit.data, buffer, length);
	RecvFlush();
	status = SendBuffer(transmit.buffer, 9+transmit.len);
	status = RecvBuffer(&receive);
	if (status==0)
	{
		*len = receive.len+1;
		memcpy(buffer, receive.data, *len);
	}	
	else
		*len = 0;

	return (status);
}

int TIMS_SPIDeviceIO(U8 addr, U8 buffer[], int* len)
{
	TIMS_packet transmit={0};
	TIMS_packet receive={0};
	int status;
	int count;
	int length = *len;
	
	if (length<1||length>255)
		return 5247;	// invalid length

	transmit.to 	= TIMS_ID;
	transmit.from	= HOST_ID;
	transmit.ctrl	= BYTE_SWAP(SPI_DEVICE_IO);
	transmit.len	= length;
	transmit.data[0] = addr;
	memcpy(transmit.data+1, buffer, length);
	RecvFlush();
	status = SendBuffer(transmit.buffer, 9+transmit.len);
	status = RecvBuffer(&receive);
	if (status==0)
	{
		*len = receive.len+1;
		memcpy(buffer, receive.data, *len);
	}
	else
		*len = 0;
		
	return (status);
}



// TIMS transmit/receive primitives //

static int SendBuffer(char* buffer, int nchars)
{
	int status;
	int count;
	int lrc = 0xAA;
	int i;
	
	if (handle==0)
		return -1; 

	if (USE_LRC)
	{
	// calculate LRC
		for (i=0; i<nchars; i++)
		{
			lrc+= buffer[i];
			lrc<<=1;
			if (lrc&0x0100)
				lrc++;	// handle carry around
			lrc&=0x00FF;
		}
		buffer[nchars++]=(U8)(lrc);	// append LRC
	}
	status = FT_Write(handle, buffer, nchars, &count);
	Show(buffer, count);
	return status;
}


static int RecvBuffer(TIMS_packet* receive)
{
	int timeout=TIMEOUT_LOOPS;
	int count;
	int nchars=0;	// total received
	int lrc=0xAA;	// seed
	U8 rxLrc;
	int i;

	if (handle==0)
		return -1; 

	// get first 8 bytes body of rx message
	while(timeout-- >0)
	{
		status = FT_GetQueueStatus(handle, &count);
		if (count>=8)
		{
			status = FT_Read(handle, receive->buffer, 8, &count);	// rd 8 bytes
			nchars=count;
			break;
		}
	}
	if(timeout<=0)
	{
		Show(receive->buffer, nchars);	// debug
		return (TIMS_TIMEOUTBODY);
	}

	// get remaining bytes data of rx message
	while(timeout-- >0)
	{
		status = FT_GetQueueStatus(handle, &count);
		if (count>=receive->len+1)
		{
			status = FT_Read(handle, &receive->buffer[nchars], receive->len+1, &count);
			nchars+=count;
			break;
		}
	}
	if(timeout<=0)
	{
		Show(receive->buffer, nchars);	// debug
		return (TIMS_TIMEOUTDATA);
	}

	
	if (USE_LRC)
	{
		// calculate LRC
		for (i=0; i<nchars; i++)
		{
			lrc+= receive->buffer[i];
			lrc<<=1;
			if (lrc&0x0100)
				lrc++;	// handle carry around
			lrc&=0x00FF;
		}

		// get the last (LRC) byte from the rx message
		while(timeout-- >0)
		{
			status = FT_GetQueueStatus(handle, &count);
			if (count>=1)
			{
				status = FT_Read(handle, &rxLrc, 1, &count);
				// add lrc to buffer for debugging
				receive->buffer[nchars++]=rxLrc;
				break;
			}
		}
		if(timeout<=0)
		{
			Show(receive->buffer, nchars);	// debug
			return (TIMS_TIMEOUTLRC);
		}
	}
	
	Show(receive->buffer, nchars);
	receive->ctrl   = BYTE_SWAP(receive->ctrl);
	receive->status = BYTE_SWAP(receive->status);

	// compare LRC with calculated value to validate packet
	if (USE_LRC && rxLrc!=lrc)
		receive->status = TIMS_CHECKSUMERR;	// lrc error - what is errorcode?
	
	return receive->status;
}

static void RecvFlush(void)
{
	char rxbuf[256];
	int count;

	status = FT_GetQueueStatus(handle, &count);
	if (count>0)
		status = FT_Read(handle,rxbuf,count,&count); // discard
	if (count>0)
		Show(rxbuf,count);	// debug - shouldn't occur
}


// DEBUGGING //

static void Show(char buffer[], int count)
{
	int i;
#if DEBUGGING
	for (i=0; i<count; i++)
		DebugPrintf("%02x ",buffer[i]&0xFF);
		
	DebugPrintf("\n");
#endif
}

